const express = require('express');
const path = require('path');
const cors = require('cors');
const dotenv = require('dotenv');
const { GoogleGenAI, Type } = require('@google/genai');

dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;

app.use(cors());
app.use(express.json({ limit: '50mb' }));

// Serve static files from the Vite build directory
app.use(express.static(path.join(__dirname, 'dist')));

// --- API ROUTES ---

/**
 * Proxy for Gemini AI Jewelry Generation
 */
app.post('/api/generate-jewelry', async (req, res) => {
    const { request } = req.body;
    const apiKey = process.env.GEMINI_API_KEY || process.env.VITE_GEMINI_API_KEY;

    if (!apiKey) {
        return res.status(500).json({ error: 'Gemini API Key is not configured on the server.' });
    }

    try {
        const ai = new GoogleGenAI(apiKey);
        const model = ai.getGenerativeModel({ model: 'gemini-2.5-flash' });
        const imageModel = ai.getGenerativeModel({ model: 'gemini-2.5-flash-image' });

        // 1. Generate Master Prompt
        const designBrief = `
      Item: ${request.type}
      Material: ${request.material}
      Gemstone: ${request.gemstone}
      Style/Aesthetic: ${request.style}
      User Description: ${request.description}
    `;

        const masterPromptResp = await model.generateContent(`
      You are a master jeweler and 3D artist. Create a "Master Visual Specification" for a jewelry piece based on this brief: "${designBrief}".
      Adapt the design language to the requested style (${request.style}).
      Describe the piece in extreme physical detail (geometry, setting, texture, gemstone placement).
      Output a single paragraph around 100 words. Do not use markdown.
    `);
        const masterPrompt = masterPromptResp.response.text().trim();

        // 2. Generate Images (Sequential with delays to avoid 429)
        const modelPrompt = `Macro product photography, 3D render. Subject: ${masterPrompt}. View: Isometric view floating in a void. Lighting: Studio lighting. Background: Solid matte black. Style: Hyper-realistic, 8k luxury catalog.`;

        const bodyPart = getBodyPart(request.type);
        const onBodyPrompt = `High-fashion editorial photography. Subject: A professional model wearing: ${masterPrompt} on their ${bodyPart}. Framing: Close-up. Style: Vogue/GQ style.`;

        const modelImageResp = await imageModel.generateContent({
            contents: [{ role: 'user', parts: [{ text: modelPrompt }] }],
            generationConfig: { temperature: 0.2 }
        });

        // Simulate delay for safety
        await new Promise(r => setTimeout(r, 3000));

        const onBodyImageResp = await imageModel.generateContent({
            contents: [{ role: 'user', parts: [{ text: onBodyPrompt }] }],
            generationConfig: { temperature: 0.2 }
        });

        const extractImage = (resp) => {
            const parts = resp.response.candidates?.[0]?.content?.parts || [];
            for (const part of parts) {
                if (part.inlineData) return `data:image/png;base64,${part.inlineData.data}`;
            }
            return '';
        };

        const modelImage = extractImage(modelImageResp);
        const onBodyImage = extractImage(onBodyImageResp);

        // 3. Verification
        const verificationPrompt = `Verify if Image 2 matches Image 1 (consistent shape, metal, stones). Confirm correct placement on ${bodyPart}. Return JSON: { "score": 0-100, "analysis": string, "isConsistent": boolean }`;

        const cleanBase64 = (b64) => b64.replace(/^data:image\/\w+;base64,/, "");

        const verificationResp = await model.generateContent({
            contents: [{
                role: 'user',
                parts: [
                    { inlineData: { mimeType: 'image/png', data: cleanBase64(modelImage) } },
                    { inlineData: { mimeType: 'image/png', data: cleanBase64(onBodyImage) } },
                    { text: verificationPrompt }
                ]
            }],
            generationConfig: {
                responseMimeType: "application/json"
            }
        });

        const verification = JSON.parse(verificationResp.response.text());

        res.json({
            masterPrompt,
            modelImage,
            onBodyImage,
            verification
        });

    } catch (error) {
        console.error('Gemini Proxy Error:', error);
        res.status(500).json({ error: error.message });
    }
});

/**
 * Proxy for Ziina Payment Intent
 */
app.post('/api/create-payment', async (req, res) => {
    const { amount, tokens, successUrl, cancelUrl } = req.body;
    const apiKey = process.env.ZIINA_API_KEY || process.env.VITE_ZIINA_API_KEY;

    if (!apiKey) {
        return res.status(500).json({ error: 'Ziina API Key is not configured on the server.' });
    }

    try {
        const response = await fetch('https://api-v2.ziina.com/api/payment_intent', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`,
            },
            body: JSON.stringify({
                amount: amount * 100, // Convert to base units
                currency: 'AED',
                success_url: `${successUrl}?payment_status=success&tokens=${tokens}`,
                cancel_url: `${cancelUrl}?payment_status=cancelled`,
                message: `Purchase ${tokens} Dahab Tokens`
            }),
        });

        const data = await response.json();
        if (!response.ok) throw new Error(data.message || 'Ziina API error');

        res.json(data);
    } catch (error) {
        console.error('Ziina Proxy Error:', error);
        res.status(500).json({ error: error.message });
    }
});

// Fallback for SPA routing: serve index.html for any unknown routes
app.get('*', (req, res) => {
    res.sendFile(path.join(__dirname, 'dist', 'index.html'));
});

// Helper functions (copied from service for server use)
function getBodyPart(type) {
    const parts = { 'Ring': 'finger', 'Necklace': 'neck', 'Earrings': 'ear', 'Bracelet': 'wrist', 'Brooch': 'lapel', 'Tiara': 'head', 'Cufflinks': 'shirt cuff', 'Pendant': 'chest', 'Watch': 'wrist', 'Full Jewelry Set': 'neck and ears' };
    return parts[type] || 'body';
}

app.listen(PORT, () => {
    console.log(`Eden Style Server running on port ${PORT}`);
});
